<?php
require 'vendor/autoload.php';
$config = require __DIR__ . '/config.php';

$type = strtolower($config['ServerFiles'] ?? 'louis');

// ✅ Устанавливаем Stripe ключи
\Stripe\Stripe::setApiKey($config['stripe_secret_key']);
$endpoint_secret = $config['stripe_webhook_key'];

// ✅ Подключение к MSSQL
try {
    $dsn = "dblib:host={$config['mssql_host']};dbname={$config['mssql_db']}";
    $pdo = new PDO($dsn, $config['mssql_user'], $config['mssql_password']);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    file_put_contents("stripe-webhook.log", "[❌ MSSQL ERROR] " . $e->getMessage() . "\n", FILE_APPEND);
    http_response_code(500);
    exit();
}

// 🔁 Получение Stripe события
$payload = @file_get_contents('php://input');
$sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';
$event = null;

try {
    $event = \Stripe\Webhook::constructEvent($payload, $sig_header, $endpoint_secret);
} catch (\Exception $e) {
    http_response_code(400);
    file_put_contents("stripe-webhook.log", "[❌ Invalid Stripe event] " . $e->getMessage() . "\n", FILE_APPEND);
    exit();
}

// ✅ Обработка оплаты
if ($event->type === 'checkout.session.completed') {
    $session = $event->data->object;

    if ($session->payment_status !== 'paid') {
        file_put_contents("stripe-webhook.log", "[⏳ Not paid] " . json_encode($session) . "\n", FILE_APPEND);
        http_response_code(200);
        exit();
    }

    $charname = $session->metadata->charname ?? '';
    $currency = $session->metadata->currency ?? 'WCoinC';
    $amount   = (int)($session->metadata->amount ?? 0);
    $eur      = $session->amount_total / 100;

    $columnMap = [
        'WCoinC'      => 'WCoinC',
        'WCoinP'      => 'WCoinP',
        'GoblinPoint' => 'GoblinPoint'
    ];
    $column = $columnMap[$currency] ?? null;

    // 🧠 Выбираем нужную таблицу в зависимости от ServerFiles
    switch ($type) {
        case 'ignc':
            $table = 'T_InGameShop_Point';
            break;
        case 'ssemu':
            $table = null; // ❌ нет таблицы
            break;
        default:
            $table = 'CashShopData';
            break;
    }

    if (!$column || $amount <= 0 || $charname === '') {
        file_put_contents("stripe-webhook.log", "[⚠️ Invalid data] $charname | $currency | $amount\n", FILE_APPEND);
        http_response_code(400);
        exit;
    }

    if (!$table) {
        file_put_contents("stripe-webhook.log", "[⚠️ Skipped] No cashshop table for $type\n", FILE_APPEND);
        http_response_code(200);
        exit;
    }

    try {
        $sql = "
            UPDATE $table
            SET $column = ISNULL($column, 0) + :amount
            WHERE AccountID = (
                SELECT AccountID FROM Character WHERE Name = :charname
            )
        ";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            'amount'   => $amount,
            'charname' => $charname
        ]);

        $log = sprintf(
            "[%s] ✅ Stripe payment: %s | %s +%d | €%.2f\n",
            date("Y-m-d H:i:s"),
            $charname,
            $currency,
            $amount,
            $eur
        );
        file_put_contents("stripe-webhook.log", $log, FILE_APPEND);

    } catch (PDOException $e) {
        file_put_contents("stripe-webhook.log", "[❌ DB Error] " . $e->getMessage() . "\n", FILE_APPEND);
        http_response_code(500);
        exit();
    }
}

http_response_code(200);
